function [] = convertmodfiles(infilename, policy_problem, instrument1, instrument2)

% INPUTS:
% 
%   1) infilename: a string, giving the name of the .mod file you want to edit (infilename.mod).
%   2) policy_problem: a string, either 'nash', 'ramsey', or 'one_agent_ramsey'. 
%   3) instrument1: a string, giving the policy instrument of the first agent. 
%                   If you want to compute the one-agent Ramsey game, then
%                   this should be the policy instrument of the agent following Ramsey policy.
%   4) instrument2: a string, the policy instrument of the second agent, 
%                   if computing the two-agent Ramsey or Nash problems.
%                   If computing the one-agent Ramsey policy, this should
%                   be '1' or '2', representing the agent which is using Ramsey policy.
%
%
% README: convertmodfiles.m
% 
% This program generates all files for computing optimal policy in a two-agent
% DSGE model. Before running convertmodfiles.m, edit the input infilename.mod file as follows:
% 
%   1) Add variables Util1 and Util2 to the var definition block. In the
%   model block, define Util1 and Util2 as each player's period-utility
%   function. These equations should take the form 'Util1 = ...;' and 
%   'Util2 = ...;'. 
%  
%   2) Break the 'var' block into two blocks. The first block should
%   contain Util1, Util2, and all endogenous variables. The second block
%   should contain all "exogenous variables", which for htis purpose are
%   defined as those that describe the shock processes for the model,
%   rather than their stochastic innovations included in the varexo
%   definition block. Insert the line '// Endogenous variables' or
%   '//Exogenous variables' before each block, as appropriate.
% 
%   3) If the parameter values are set in the model file, move them in a
%   separate script in the same folder under the name
%   'infilename_paramfile.m.' The structure of the file name is significant;
%   the first part needs to match the string passed to infilename, and the
%   second part must be '_paramfile.m'. In addition to the original
%   parameters, add three new paramters to infilename_paramfile.m and to
%   the parameter definition block in infilename.mod:
%       - nbeta, the discount factor in each player's welfare function(s),
%       - omega_welf1, the weight of player 1 in the joint welfare function
%       under the cooperative Ramsey policy, and
%       - omega_welf2, the weight of player 2 in the joint welfare function
%       under the cooperative Ramsey policy, equaling 1 - omega_welf1.
%
%   4) In the model block, in the line immediately preceding the policy
%   rule for each player, insert the comment '// Policy Rule, agent 1' or
%   '//Policy Rule, agent 2', as appropriate. (N.B.: the capitalization of
%   the comments is significant. Also, the second policy rule comment is
%   not needed if policy_problem = 'one_agent_ramsey'.
%
%   5) If the steady-state values for the original N endogenous variables
%   are set in the initval block, move that section to a script in the same
%   folder under the name infilename_ss_defs.m. (As detailed above, the
%   structure of the name is significant.) The script infilename_ss_defs.m
%   should also include the steady-state values for the additional
%   variables Util1 and Util2.
%
%   6) Collect the equations describing the paths of exogenous variables at
%   the end of the model block, after all the structural equations that
%   include endogenous variables.
%
% Then, create a function in the same folder as infilename.mod named
% infilename_steadystate.m. Dynare will call upon this program to compute
% the steady state of the model. The structure should mimic that of
% example_steadystate.m, which is also containted in the toolbox.
% 
%
% PROCESS:
% 
% After setting file paths, get_aux.m is called, which:
% 	- replaces lagged endogenous variables in the model block with
%   auxiliary variables, which are also inserted under the 'var' block as
%   endogenous variables. Given endogenous variables var_1, ..., var_k
%   entering the structural equations or utility functions with lagged
%   values, get_aux.m adds var_1lag, ..., var_klag to the end of the
%   block of endogenous variables in the var block, and adds the
%   equations var_1lag = var_1(-1); ... var_klag = var_k(-1); in the
%   model block. This step avoids complications that arise from the
%   optimal policy equations containing endogenous variables that are
%   lagged more than one period.
%   - given policy_problem, adds the appropriate policy variables (instr1
%   and, if necessary, instr2), parameters, and equations to the model to
%   set up the equilibrium conditions of the specified policy problem.
%   The new model file is temporarily saved as infilename_aux.mod.
%   - edits infilename_paramfile.m, infilename_steadystate.m, and
%   infilename_ss_defs.m to account for the auxiliary and policy variables,
%   parameters, and equations. The new files reflecting the changes are
%   temporarily saved as infilename_aux_paramfile.m,
%   infilename_aux_steadystate.m, and infilename_aux_ss_defs.m,
%   respectively.
%
% Next, depending on the choice of policy_problem, get_nash.m or
% get_ramsey.m is called. These programs take as an input
% infilename_aux.mod and output infilename_'policy_problem'.mod (with the
% exception of when policy_problem = 'one_agent_ramsey', in which case the
% output Dynare model file is infilename_ramsey_agent_'instrument2'.mod,
% where instrument2 represents the player using optimal policy. These
% naming conventions for the one-agent Ramsey problem hold for all
% auxiliary programs as well, but for simplicity all descriptions below
% will consider infilename_'policy_problem'.mod). The output .mod file 
% contains the first-order conditions of the players and removes
% the original policy rules. In addition, these programs create the script
% infilename_'policy_problem'_lmss.m, which will be called in the new
% _steadystate.m file to calculate the steady-state values of the
% Lagrangian multipliers.
%
% Then, the file make_ss_'policy_problem'.m is called, which creates
% auxiliary files infilename_'policy_problem'_steadystate.m, _ss_defs.m,
% and _paramfile.m. Among other features, these three auxiliary files
% differ from their original version as follows:
%   - As we have introduced additional endogenous variables, the
%   steady-state values of the existing endogenous variables may have
%   changed, and the steady-state values of the new endogenous variables
%   are unspecified. infilename_'policy_problem'_steadystate.m uses the
%   values provided by infilename_steadystate.m (for the original
%   endogenous variables) and infilename_'policy_problem'_lmss.m (for the
%   Lagrangian multipliers) as an initial guess to find the new
%   steady-state values which are required to solve
%   infilename_'policy_problem'_static.m.
%   - infilename_'policy_problem'_ss_defs.m has been augmented to set an
%   initial guess for steady-state values for the Lagrangian multipliers.
%
% Finally, we delete the temporary files co run test_for_ss.m, which tests whether the steady-state value
% from the initial guess in infilename_'policy_problem'_ss_defs.m solves
% infilename_'policy_problem'.mod. If not, test_for_ss finds a better guess
% for the steady-state value using simulated annealing, and then replaces
% the contents of infilename_'policy_problem'_ss_defs.m with that guess.
% 
% OUTPUTS:
%   
%   1) infilename_'policy_problem'.mod
%   2) infilename_'policy_problem'_paramfile.m
%   3) infilename_'policy_problem'_ss_defs.m
%   4) infilename_'policy_problem'_steadystate.m 
%% Set paths

restoredefaultpath 
setpathdynare4

% Removing '.mod' if passed as part of infilename.
if strncmp(infilename(length(infilename)-3:length(infilename)),'.mod',4)
    infilename = infilename(1:length(infilename)-4);
end;

%% Choose type of file to convert

if strcmp(policy_problem,'nash')    
    get_aux(infilename,'nash',instrument1,instrument2)
%    outfilename = [infilename,'_nash'];
    outfilename = [infilename,'_nash','_',instrument1,'_',instrument2];
    infilename = [infilename,'_aux'];
    get_nash;
    make_ss_nash;
elseif strcmp(policy_problem,'ramsey')    
    get_aux(infilename,'ramsey',instrument1,instrument2)
%    outfilename = [infilename,'_ramsey'];
    outfilename = [infilename,'_ramsey','_',instrument1,'_',instrument2];
    infilename = [infilename,'_aux'];
    get_ramsey;
    make_ss_ramsey;
elseif strcmp(policy_problem,'one_agent_ramsey')
    ramsey_agent_string = num2str(instrument2);
    get_aux(infilename,'one_agent_ramsey',instrument1,ramsey_agent_string);
%    outfilename = [infilename,'_ramsey_agent','_',ramsey_agent_string];
    outfilename = [infilename,'_ramsey_agent','_',instrument1,'_',ramsey_agent_string];
    infilename = [infilename,'_agent_',ramsey_agent_string,'_aux'];
    get_ramsey;
    make_ss_one_agent_ramsey;
else
    error('Please input an acceptable policy problem.');
end;

fclose('all');

delete([infilename,'*']);

